#ifndef LINT
static char *rcsid="$Id: spf.c 485 2009-03-29 22:48:00Z crosser $";
#endif

/*
	WHAT IS IT:
		modularized contentfilter for Zmailer
	COPYRIGHT:
		(c) 2005 Eugene G. Crosser <crosser@average.org>
	LICENSE:
		The same set as apply to Zmailer code
*/

#include "config.h"

#include <sys/types.h>
#ifdef STDC_HEADERS
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
#endif

#include <zmscanner.h>

#include "report.h"
#include "libspf_any.h"
#include "spf_config.h"

#define min(x,y) ((x<y)?x:y)

static struct _levels {
	char *txt;
	int num;
} levels[] = {
	{"invalid",	5},
	{"neutral",	4},
	{"pass",	5},
	{"fail",	1},
	{"softfail",	2},
	{"none",	3},
	{"error",	5},
	{"unknown",	5},
	{NULL,		0}
};

static int
numericlevel(char *level)
{
	int num=5;
	int i;

	for (i=0;levels[i].txt;i++) {
		if (strcmp(levels[i].txt,level) == 0) {
			num=levels[i].num;
			break;
		}
	}
	return num;
}

static void *svcfg=NULL;

static int
nul_setup(void **cfgp) { (*cfgp)=svcfg; return 0; }
static void
nul_term(void **cfgp) { return; }

static int
spf_setup(void **cfgp)
{
	DPRINT(("spf_setup starting\n"));
	if (((*cfgp)=spf_newconfig()) == NULL) {
		ERRLOG((LOG_ERR,"spf: cannot allocate space for cfg object"));
		return -1;
	}
	ERRLOG((LOG_INFO,"spf: %s (%s) OK",VERSION,rcsid));
	svcfg=*cfgp;
	return 0;
}

static void
spf_term(void **cfgp)
{
	DPRINT(("spf_term entered\n"));
	spf_cleanconfig(*cfgp);
	(*cfgp)=NULL;
}

#define SPF_PARM_OBJECT "spf_parm_object"

typedef struct _spf_parm {
	char	magic[4];
	char 	*peeraddr;
	spf_state_t spfstate;
} spf_parm_t;

static void
spf_parm_destruct(void *object)
{
	spf_parm_t *parm=(spf_parm_t *)object;

	DPRINT(("destructing spf_parm object at %p\n",parm));
	if (parm->peeraddr) free(parm->peeraddr);
	spf_freestate(parm->spfstate);
	free(parm);
}

static int
spf_scan_peerdata(char *stage,int depth,slab_t data,varpool_t vp,void *priv)
{
	cfg_t *cfg=(cfg_t*)priv;
	spf_parm_t *parm;
	char caddr[64];
	char *p,*q;
	int secondel,bopen,v6;

	DPRINT(("spf_scan_peerdata enter\n"));

	/* this is the first processor for a message; create parm object */
	if ((parm=(spf_parm_t *)malloc(sizeof(spf_parm_t))) == NULL) {
		ERRLOG((LOG_ERR,"spf: cannot allocate space for parm object"));
		return ZMSCAN_CONTINUE;
	}
	strcpy(parm->magic,"spf"); /* four bytes including terminator */
	parm->peeraddr=NULL;
	if ((parm->spfstate=spf_newstate(cfg->global)) == NULL) {
		ERRLOG((LOG_ERR,"spf: cannot create spfsate object"));
		return ZMSCAN_CONTINUE;
	}
	vp_set(vp,SPF_PARM_OBJECT,(void *)parm,spf_parm_destruct);

	bopen=0;
	secondel=0;
	v6=0;
	q=caddr;
	for (p=data.beg;p<data.end;p++) switch (*p) {
	case ' ': secondel=1;
	case '[': if (secondel) bopen=1; break;
	case ']': if (secondel) bopen=0; break;
	default:
		if (bopen && (q < caddr+sizeof(caddr))) {
			*q++=*p;
			if (*p == ':') v6=1;
		}
		break;
	}
	*q='\0';
	if (caddr[0] == '\0') {
		ERRLOG((LOG_ERR,"spf: empty address\n"));
		return ZMSCAN_CONTINUE;
	}
	if ((parm->peeraddr=(char*)malloc(strlen(caddr)+1)))
		strcpy(parm->peeraddr,caddr);
	DPRINT(("spf pass peerip: \"%s\", v6=%d\n",caddr,v6));
	if (spf_peerip(cfg->global,parm->spfstate,caddr,v6)) {
		ERRLOG((LOG_ERR,"spf: peerip set failed\n"));
		return ZMSCAN_CONTINUE;
	}
	return ZMSCAN_CONTINUE;
}

static int
spf_scan_helo(char *stage,int depth,slab_t data,varpool_t vp,void *priv)
{
	cfg_t *cfg=(cfg_t*)priv;
	spf_parm_t *parm=(spf_parm_t *)vp_get(vp,SPF_PARM_OBJECT);
	char buf[128];
	int len;

	DPRINT(("spf_scan_helo enter, parm at %p\n",parm));
	if ((parm == NULL) || (strcmp(parm->magic,"spf") != 0)) {
		ERRLOG((LOG_ERR,"spf: parm object missing or invalid"));
		return ZMSCAN_CONTINUE;
	}
	len=min(sizeof(buf)-1,slab_size(data));
	strncpy(buf,data.beg,len);
	buf[len]='\0';
	DPRINT(("spf pass helo: \"%s\"\n",buf));
	if (spf_helo(cfg->global,parm->spfstate,buf)) {
		ERRLOG((LOG_ERR,"spf: helo set failed\n"));
		return ZMSCAN_CONTINUE;
	}
	return ZMSCAN_CONTINUE;
}

static int
spf_scan_envfrom(char *stage,int depth,slab_t data,varpool_t vp,void *priv)
{
	cfg_t *cfg=(cfg_t*)priv;
	spf_parm_t *parm=(spf_parm_t *)vp_get(vp,SPF_PARM_OBJECT);
	char buf[128];
	int len;
	char *level,*com,*hdr,*auth_authen;
	int numlevel;
	char ans[256];

	DPRINT(("spf_scan_envfrom enter, parm at %p\n",parm));
	if ((parm == NULL) || (strcmp(parm->magic,"spf") != 0)) {
		ERRLOG((LOG_ERR,"spf: parm object missing or invalid"));
		return ZMSCAN_CONTINUE;
	}
	len=min(sizeof(buf)-1,slab_size(data));
	strncpy(buf,data.beg,len);
	buf[len]='\0';
	DPRINT(("spf pass envfrom: \"%s\"\n",buf));
	if (spf_from(cfg->global,parm->spfstate,buf)) {
		ERRLOG((LOG_ERR,"spf: envfrom set failed\n"));
		return ZMSCAN_CONTINUE;
	}
	if (spf_result(cfg->global,parm->spfstate,&level,&com,&hdr)) {
		ERRLOG((LOG_ERR,"spf: spf result call failed\n"));
		return ZMSCAN_CONTINUE;
	}
	DPRINT(("spf result: %s\n",level));
	DPRINT(("spf comment: %s\n",com?com:"no comment"));
	DPRINT(("spf header: %s\n",hdr?hdr:"<null>"));
	vp_set_str(vp,"spf_level",level?level:"null");
	vp_set_str(vp,"spf_comment",com?com:"null");
	if (iprangelist_lookup(cfg->whitelist,rl_type_unknown,
						parm->peeraddr)) {
		ERRLOG((LOG_INFO,"spf: Peer IP %s in whitelist, bypass check "
				"(result was %s)",parm->peeraddr,level));
		return ZMSCAN_CONTINUE;
	}
	auth_authen=vp_get_str(vp,"smfi_auth_authen");
	if (auth_authen) {
		ERRLOG((LOG_INFO,"spf: User %s authenticated, bypass check "
				"(result was %s)",auth_authen,level));
		return ZMSCAN_CONTINUE;
	}
	if (cfg->addspfheader && hdr)
		vp_insert_strl(vp,VP_ADDHEADERS_STRL,hdr);
	numlevel=numericlevel(level);
#if 1
	DPRINT(("spf level=%s (%d), threshold=%d, will %s\n",
		level,numlevel,cfg->threshold,
		(numlevel < cfg->threshold)?"block":"pass"));
#else
	ERRLOG((LOG_INFO,"spf level=%s (%d), threshold=%d, will %s",
		level,numlevel,cfg->threshold,
		(numlevel < cfg->threshold)?"block":"pass"));
#endif
	if (numlevel < cfg->threshold) {
		snprintf(ans,sizeof(ans),
			"-1 553 5.7.1 SPF %s. %s",level,com?com:"(no comment)");
		vp_set_str(vp,VP_ANSWER_STR,ans);
		return ZMSCAN_STOP;
	} else return ZMSCAN_CONTINUE;
}

ZMS_MODINIT_FUNC
{
	ZMS_CLAIM_STAGE("peerdata","spf",spf_setup,spf_term,spf_scan_peerdata);
	ZMS_CLAIM_STAGE("helo","spf",nul_setup,nul_term,spf_scan_helo);
	ZMS_CLAIM_STAGE("envfrom","spf",nul_setup,nul_term,spf_scan_envfrom);
}
