#ifndef LINT
static char *rcsid="$Id: clamav_refreshdb.c 483 2009-03-29 16:41:19Z crosser $";
#endif

/*
	WHAT IS IT:
		modularized contentfilter for Zmailer
	COPYRIGHT:
		(c) 2003-2005 Eugene G. Crosser <crosser@average.org>
	LICENSE:
		The same set as apply to Zmailer code
*/

#include "config.h"

#include <sys/types.h>
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#ifdef STDC_HEADERS
# include <stdio.h>
# include <stdlib.h>
# include <string.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#if defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR)
# include <pthread.h>
#endif /* defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR) */

#include <zmscanner.h>
#include <clamav.h>

#include "clamav_config.h"
#include "report.h"

#ifdef HAVE_GETTIMEOFDAY
static double tdiff(struct timeval *t1,struct timeval *t2)
{
	double mdiff=(t2->tv_usec-t1->tv_usec)/1e6;
	return mdiff+(t2->tv_sec-t1->tv_sec);
}
#else
static double tdiff(time_t *t1,time_t *t2)
{
	return (double)1e0*(*t2-*t1);
}
#endif

static struct cl_engine *
loaddb(clamav_cfg_t *cfg)
{
	struct cl_engine *newroot=NULL;
	int rc=0;
	unsigned int virnum=0;
#ifdef HAVE_GETTIMEOFDAY
	struct timeval t1,t2,t3;
#else
	time_t t1,t2,t3;
#endif

	DPRINT(("loaddb called\n"));
#ifdef HAVE_GETTIMEOFDAY
	(void)gettimeofday(&t1,NULL);
#else
	(void)time(&t1);
#endif
#ifdef HAVE_CL_INIT
	newroot=cl_engine_new();
	if (newroot == NULL) {
		ERRLOG((LOG_ERR,"clamav: cl_engine_new()"));
		return NULL;
	}
	if (cfg->settings) {
		rc=cl_engine_settings_apply(newroot,cfg->settings);
	}
	if (rc) {
		ERRLOG((LOG_ERR,"clamav: settings_apply() error: %s, "
				"will use defaults",
				cl_strerror(rc)));
	}
	rc=cl_load(cfg->finaldbdir,newroot,&virnum,cfg->dbopts);
#else
# ifdef HAVE_CL_LOAD
	rc=cl_load(cfg->finaldbdir,&newroot,&virnum,cfg->dbopts);
# else
	rc=cl_loaddbdir(cfg->finaldbdir,&newroot,&virnum);
# endif
#endif
	DPRINT(("loaddb: load rc=%d\n",rc));
	if (rc) {
		ERRLOG((LOG_ERR,"clamav: loaddbdir(\"%s\",...) error: %s",
					cfg->finaldbdir,cl_strerror(rc)));
		return NULL;
	}
#ifdef HAVE_GETTIMEOFDAY
	(void)gettimeofday(&t2,NULL);
#else
	(void)time(&t2);
#endif
#ifdef HAVE_CL_INIT
	rc=cl_engine_compile(newroot);
#else
	rc=cl_build(newroot);
#endif
	DPRINT(("loaddb: build rc=%d\n",rc));
	if (rc) {
		ERRLOG((LOG_ERR,"clamav: build error: %s",
					cl_strerror(rc)));
		return NULL;
	}
#ifdef HAVE_GETTIMEOFDAY
	(void)gettimeofday(&t3,NULL);
#else
	(void)time(&t3);
#endif
	
	ERRLOG((LOG_INFO,"clamav: loaded %d virus signatures "
			"(load:%7.3f, build:%7.3f sec)",
			virnum,tdiff(&t1,&t2),tdiff(&t2,&t3)));
	return newroot;
}

static int
replacedb(clamav_cfg_t *cfg, struct cl_engine *newroot)
{
	if (newroot == NULL) return 1;
	if (cfg->root) {
#ifdef HAVE_CL_INIT
		cl_engine_free(cfg->root);
#else
		cl_free(cfg->root);
#endif
	}
	cfg->root=newroot;
	return 0;
}

#if defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR)

static int count=0;
static pthread_cond_t count_cond=PTHREAD_COND_INITIALIZER;
static pthread_mutex_t count_mutex=PTHREAD_MUTEX_INITIALIZER;

static int standby=0;
static pthread_cond_t standby_cond=PTHREAD_COND_INITIALIZER;
static pthread_mutex_t standby_mutex=PTHREAD_MUTEX_INITIALIZER;

void
pre_scan(void)
{
	pthread_mutex_lock(&standby_mutex);
	while (standby) {
		DPRINT(("pre_scan: waiting on standby flag\n"));
		pthread_cond_wait(&standby_cond,&standby_mutex);
		DPRINT(("pre_scan: standby_cond signalled\n"));
	}
	pthread_mutex_lock(&count_mutex);
	count++;
	pthread_mutex_unlock(&count_mutex);
	pthread_mutex_unlock(&standby_mutex);
}

void
post_scan(void)
{
	pthread_mutex_lock(&count_mutex);
	count--;
	if (count <= 0) {
		pthread_cond_signal(&count_cond);	/* one thread waits */
	}
	pthread_mutex_unlock(&count_mutex);
}

static void
lock_load(clamav_cfg_t *cfg)
{
	struct cl_engine *newroot=loaddb(cfg);
	if (newroot == NULL) return;
	pthread_mutex_lock(&standby_mutex);
	standby=1;
	pthread_mutex_unlock(&standby_mutex);
	DPRINT(("refresher: standby flag set, wait for count to drop\n"));
	pthread_mutex_lock(&count_mutex);
	while (count) {
		DPRINT(("refresher: waiting for count to become zero\n"));
		pthread_cond_wait(&count_cond,&count_mutex);
		DPRINT(("refresher: count_cond signalled\n"));
	}
	pthread_mutex_unlock(&count_mutex);

	DPRINT(("refresher: count dropped to zero, may refresh database\n"));
	/* do real job here */
	(void)replacedb(cfg,newroot);
	/* real job done */
	DPRINT(("refresher: refresh complete, reset standby flag\n"));

	pthread_mutex_lock(&standby_mutex);
	standby=0;
	pthread_cond_broadcast(&standby_cond);	/* many threads wait */
	pthread_mutex_unlock(&standby_mutex);
}

static void *
refresher(void *priv)
{
	clamav_cfg_t *cfg=(clamav_cfg_t *)priv;
	struct cl_stat dbstat;

	memset(&dbstat,0,sizeof(dbstat));
	cl_statinidir(cfg->finaldbdir,&dbstat);
	while (1) {
		sleep(10);
		if (cl_statchkdir(&dbstat)) {
			lock_load(cfg);
			cl_statfree(&dbstat);
			cl_statinidir(cfg->finaldbdir,&dbstat);
		}
	}
}

int
clamav_refreshdb(clamav_cfg_t *cfg)
{
	int rc;
	static pthread_t refresh_thread;
	static pthread_attr_t attr;

	DPRINT(("clamav_refreshdb thread starting\n"));
	rc=replacedb(cfg,loaddb(cfg));
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
	pthread_create(&refresh_thread,&attr,refresher,(void *)cfg);
	return rc;
}

#else /* defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR) */

void
pre_scan(void)
{}

void
post_scan(void)
{}

int
clamav_refreshdb(clamav_cfg_t *cfg)
{
	DPRINT(("clamav_refreshdb single-run\n"));
	return replacedb(cfg,loaddb(cfg));
}

#endif /* defined(_REENTRANT) && defined(HAVE_CL_STATCHKDIR) */
